<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Redirect if already logged in
if (isLoggedIn()) {
    redirect(APP_URL . '/pages/dashboard.php');
}

$pageTitle = 'Register';
$errors = [];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = sanitize($_POST['username']);
    $email = sanitize($_POST['email']);
    $password = $_POST['password'];
    $confirmPassword = $_POST['confirm_password'];
    $permalink = strtolower(sanitize($_POST['permalink']));
    
    // Validate input
    if (empty($username)) {
        $errors[] = 'Username is required';
    } elseif (strlen($username) < 3 || strlen($username) > 50) {
        $errors[] = 'Username must be between 3 and 50 characters';
    }
    
    if (empty($email)) {
        $errors[] = 'Email is required';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Invalid email format';
    }
    
    if (empty($password)) {
        $errors[] = 'Password is required';
    } elseif (strlen($password) < 6) {
        $errors[] = 'Password must be at least 6 characters';
    }
    
    if ($password !== $confirmPassword) {
        $errors[] = 'Passwords do not match';
    }
    
    if (empty($permalink)) {
        $errors[] = 'Permalink is required';
    } elseif (!preg_match('/^[a-z0-9_-]+$/', $permalink)) {
        $errors[] = 'Permalink can only contain lowercase letters, numbers, underscores, and hyphens';
    }
    
    // Check if username, email, or permalink already exists
    if (empty($errors)) {
        $existingUser = $db->selectOne("SELECT * FROM users WHERE username = ? OR email = ? OR permalink = ?", [$username, $email, $permalink]);
        
        if ($existingUser) {
            if ($existingUser['username'] === $username) {
                $errors[] = 'Username already exists';
            }
            
            if ($existingUser['email'] === $email) {
                $errors[] = 'Email already exists';
            }
            
            if ($existingUser['permalink'] === $permalink) {
                $errors[] = 'Permalink already exists';
            }
        }
    }
    
    // If no errors, create user account
    if (empty($errors)) {
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        
        $userData = [
            'username' => $username,
            'email' => $email,
            'password' => $hashedPassword,
            'permalink' => $permalink
        ];
        
        $userId = $db->insert('users', $userData);
        
        if ($userId) {
            // Set flash message and redirect to login page
            setFlashMessage('success', 'Registration successful! You can now log in.');
            redirect(APP_URL . '/pages/login.php');
        } else {
            $errors[] = 'An error occurred. Please try again.';
        }
    }
}

include '../templates/header.php';
?>

<div class="row justify-content-center">
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h3 class="text-center">Register</h3>
            </div>
            <div class="card-body">
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo $error; ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <form method="POST">
                    <div class="mb-3">
                        <label for="username" class="form-label">Username</label>
                        <input type="text" id="username" name="username" class="form-control" value="<?php echo isset($_POST['username']) ? sanitize($_POST['username']) : ''; ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="email" class="form-label">Email</label>
                        <input type="email" id="email" name="email" class="form-control" value="<?php echo isset($_POST['email']) ? sanitize($_POST['email']) : ''; ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="permalink" class="form-label">Permalink</label>
                        <div class="input-group">
                            <span class="input-group-text"><?php echo APP_URL; ?>/</span>
                            <input type="text" id="permalink" name="permalink" class="form-control" value="<?php echo isset($_POST['permalink']) ? sanitize($_POST['permalink']) : ''; ?>" required>
                        </div>
                        <small class="form-text text-muted">This will be your unique URL (e.g., yourname)</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="password" class="form-label">Password</label>
                        <input type="password" id="password" name="password" class="form-control" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="confirm_password" class="form-label">Confirm Password</label>
                        <input type="password" id="confirm_password" name="confirm_password" class="form-control" required>
                    </div>
                    
                    <div class="d-grid">
                        <button type="submit" class="btn btn-primary">Register</button>
                    </div>
                </form>
                
                <div class="mt-3 text-center">
                    <p>Already have an account? <a href="<?php echo APP_URL; ?>/pages/login.php">Login here</a></p>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include '../templates/footer.php'; ?>