<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Redirect if not logged in
if (!isLoggedIn()) {
    setFlashMessage('danger', 'Please log in to access this page');
    redirect(APP_URL . '/pages/login.php');
}

$pageTitle = 'Edit Profile';
$userId = $_SESSION['user_id'];
$errors = [];
$success = '';

// Get user data
$user = getUserById($userId);

// Get user's social links
$socialLinks = getSocialLinksByUserId($userId);
$socialLinksByPlatform = [];
foreach ($socialLinks as $link) {
    $socialLinksByPlatform[$link['platform']] = $link;
}

// Social platforms
$socialPlatforms = ['facebook', 'twitter', 'instagram', 'tiktok', 'youtube', 'amazon', 'goodreads', 'website'];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'update_profile') {
        $username = sanitize($_POST['username']);
        $email = sanitize($_POST['email']);
        $permalink = strtolower(sanitize($_POST['permalink']));
        $currentPassword = $_POST['current_password'];
        $newPassword = $_POST['new_password'];
        $confirmPassword = $_POST['confirm_password'];
        
        // Validate input
        if (empty($username)) {
            $errors[] = 'Username is required';
        } elseif (strlen($username) < 3 || strlen($username) > 50) {
            $errors[] = 'Username must be between 3 and 50 characters';
        }
        
        if (empty($email)) {
            $errors[] = 'Email is required';
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Invalid email format';
        }
        
        if (empty($permalink)) {
            $errors[] = 'Permalink is required';
        } elseif (!preg_match('/^[a-z0-9_-]+$/', $permalink)) {
            $errors[] = 'Permalink can only contain lowercase letters, numbers, underscores, and hyphens';
        }
        
        // Check if username, email, or permalink already exists (excluding current user)
        if ($username !== $user['username'] || $email !== $user['email'] || $permalink !== $user['permalink']) {
            $existingUser = $db->selectOne(
                "SELECT * FROM users WHERE (username = ? OR email = ? OR permalink = ?) AND id != ?", 
                [$username, $email, $permalink, $userId]
            );
            
            if ($existingUser) {
                if ($existingUser['username'] === $username && $username !== $user['username']) {
                    $errors[] = 'Username already exists';
                }
                
                if ($existingUser['email'] === $email && $email !== $user['email']) {
                    $errors[] = 'Email already exists';
                }
                
                if ($existingUser['permalink'] === $permalink && $permalink !== $user['permalink']) {
                    $errors[] = 'Permalink already exists';
                }
            }
        }
        
        // Handle password change
        $updatePassword = false;
        if (!empty($newPassword)) {
            if (empty($currentPassword)) {
                $errors[] = 'Current password is required to set a new password';
            } elseif (!password_verify($currentPassword, $user['password'])) {
                $errors[] = 'Current password is incorrect';
            } elseif (strlen($newPassword) < 6) {
                $errors[] = 'New password must be at least 6 characters';
            } elseif ($newPassword !== $confirmPassword) {
                $errors[] = 'New passwords do not match';
            } else {
                $updatePassword = true;
            }
        }
        
        // Handle profile image upload
        $updateProfileImage = false;
        if (!empty($_FILES['profile_image']['name'])) {
            $uploadDir = UPLOAD_DIR;
            
            // Create upload directory if it doesn't exist
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $profileImage = uploadImage($_FILES['profile_image'], $uploadDir);
            
            if ($profileImage) {
                $updateProfileImage = true;
            } else {
                $errors[] = 'Invalid image file. Please upload a JPG, PNG, or GIF file.';
            }
        }
        
        // If no errors, update user profile
        if (empty($errors)) {
            $userData = [
                'username' => $username,
                'email' => $email,
                'permalink' => $permalink
            ];
            
            if ($updatePassword) {
                $userData['password'] = password_hash($newPassword, PASSWORD_DEFAULT);
            }
            
            if ($updateProfileImage) {
                $userData['profile_image'] = $profileImage;
            }
            
            $db->update('users', $userData, 'id = ?', [$userId]);
            
            // Update session username
            $_SESSION['username'] = $username;
            
            $success = 'Profile updated successfully';
            
            // Get updated user data
            $user = getUserById($userId);
        }
    } elseif (isset($_POST['action']) && $_POST['action'] === 'update_social') {
        // Update social links
        foreach ($socialPlatforms as $platform) {
            $url = sanitize($_POST[$platform . '_url'] ?? '');
            
            if (!empty($url)) {
                // Add http:// prefix if not present
                if (!preg_match('/^https?:\/\//i', $url)) {
                    $url = 'https://' . $url;
                }
                
                if (isset($socialLinksByPlatform[$platform])) {
                    // Update existing link
                    $db->update('social_links', ['url' => $url], 'id = ?', [$socialLinksByPlatform[$platform]['id']]);
                } else {
                    // Add new link
                    $db->insert('social_links', [
                        'user_id' => $userId,
                        'platform' => $platform,
                        'url' => $url
                    ]);
                }
            } elseif (isset($socialLinksByPlatform[$platform])) {
                // Delete existing link if URL is empty
                $db->delete('social_links', 'id = ?', [$socialLinksByPlatform[$platform]['id']]);
            }
        }
        
        $success = 'Social links updated successfully';
        
        // Refresh social links
        $socialLinks = getSocialLinksByUserId($userId);
        $socialLinksByPlatform = [];
        foreach ($socialLinks as $link) {
            $socialLinksByPlatform[$link['platform']] = $link;
        }
    }
}

include '../templates/header.php';
?>

<div class="row">
    <div class="col-md-12 mb-4">
        <h1>Edit Profile</h1>
        <p>Update your profile information and social media links.</p>
    </div>
</div>

<?php if (!empty($success)): ?>
    <div class="alert alert-success">
        <?php echo $success; ?>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-md-6">
        <div class="card mb-4">
            <div class="card-header">
                <h3>Profile Information</h3>
            </div>
            <div class="card-body">
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo $error; ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="update_profile">
                    
                    <div class="mb-3">
                        <label for="username" class="form-label">Username *</label>
                        <input type="text" id="username" name="username" class="form-control" value="<?php echo sanitize($user['username']); ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="email" class="form-label">Email *</label>
                        <input type="email" id="email" name="email" class="form-control" value="<?php echo sanitize($user['email']); ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="permalink" class="form-label">Permalink *</label>
                        <div class="input-group">
                            <span class="input-group-text"><?php echo APP_URL; ?>/</span>
                            <input type="text" id="permalink" name="permalink" class="form-control" value="<?php echo sanitize($user['permalink']); ?>" required>
                        </div>
                        <small class="form-text text-muted">This is your unique URL</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="profile_image" class="form-label">Profile Image</label>
                        <?php if (!empty($user['profile_image'])): ?>
                            <div class="mb-2">
                                <img src="<?php echo UPLOAD_URL . $user['profile_image']; ?>" alt="Profile image" class="img-thumbnail" style="max-height: 100px;">
                            </div>
                        <?php endif; ?>
                        <input type="file" id="profile_image" name="profile_image" class="form-control">
                        <small class="form-text text-muted">Upload a JPG, PNG, or GIF image</small>
                    </div>
                    
                    <hr>
                    <h4>Change Password</h4>
                    <p class="text-muted small">Leave blank to keep your current password</p>
                    
                    <div class="mb-3">
                        <label for="current_password" class="form-label">Current Password</label>
                        <input type="password" id="current_password" name="current_password" class="form-control">
                    </div>
                    
                    <div class="mb-3">
                        <label for="new_password" class="form-label">New Password</label>
                        <input type="password" id="new_password" name="new_password" class="form-control">
                    </div>
                    
                    <div class="mb-3">
                        <label for="confirm_password" class="form-label">Confirm New Password</label>
                        <input type="password" id="confirm_password" name="confirm_password" class="form-control">
                    </div>
                    
                    <div class="d-grid">
                        <button type="submit" class="btn btn-primary">Update Profile</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h3>Social Media Links</h3>
            </div>
            <div class="card-body">
                <p>Add links to your social media profiles and author pages. These will appear as icons on your book page.</p>
                
                <form method="POST">
                    <input type="hidden" name="action" value="update_social">
                    
                    <div class="mb-3">
                        <label for="website_url" class="form-label">
                            <i class="fas fa-globe"></i> Website
                        </label>
                        <input type="text" id="website_url" name="website_url" class="form-control" 
                               value="<?php echo isset($socialLinksByPlatform['website']) ? sanitize($socialLinksByPlatform['website']['url']) : ''; ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="facebook_url" class="form-label">
                            <i class="fab fa-facebook"></i> Facebook
                        </label>
                        <input type="text" id="facebook_url" name="facebook_url" class="form-control" 
                               value="<?php echo isset($socialLinksByPlatform['facebook']) ? sanitize($socialLinksByPlatform['facebook']['url']) : ''; ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="twitter_url" class="form-label">
                            <i class="fab fa-twitter"></i> Twitter
                        </label>
                        <input type="text" id="twitter_url" name="twitter_url" class="form-control" 
                               value="<?php echo isset($socialLinksByPlatform['twitter']) ? sanitize($socialLinksByPlatform['twitter']['url']) : ''; ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="instagram_url" class="form-label">
                            <i class="fab fa-instagram"></i> Instagram
                        </label>
                        <input type="text" id="instagram_url" name="instagram_url" class="form-control" 
                               value="<?php echo isset($socialLinksByPlatform['instagram']) ? sanitize($socialLinksByPlatform['instagram']['url']) : ''; ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="tiktok_url" class="form-label">
                            <i class="fab fa-tiktok"></i> TikTok
                        </label>
                        <input type="text" id="tiktok_url" name="tiktok_url" class="form-control" 
                               value="<?php echo isset($socialLinksByPlatform['tiktok']) ? sanitize($socialLinksByPlatform['tiktok']['url']) : ''; ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="youtube_url" class="form-label">
                            <i class="fab fa-youtube"></i> YouTube
                        </label>
                        <input type="text" id="youtube_url" name="youtube_url" class="form-control" 
                               value="<?php echo isset($socialLinksByPlatform['youtube']) ? sanitize($socialLinksByPlatform['youtube']['url']) : ''; ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="amazon_url" class="form-label">
                            <i class="fab fa-amazon"></i> Amazon Author Page
                        </label>
                        <input type="text" id="amazon_url" name="amazon_url" class="form-control" 
                               value="<?php echo isset($socialLinksByPlatform['amazon']) ? sanitize($socialLinksByPlatform['amazon']['url']) : ''; ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="goodreads_url" class="form-label">
                            <i class="fab fa-goodreads"></i> Goodreads
                        </label>
                        <input type="text" id="goodreads_url" name="goodreads_url" class="form-control" 
                               value="<?php echo isset($socialLinksByPlatform['goodreads']) ? sanitize($socialLinksByPlatform['goodreads']['url']) : ''; ?>">
                    </div>
                    
                    <div class="d-grid">
                        <button type="submit" class="btn btn-primary">Update Social Links</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include '../templates/footer.php'; ?>