<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Redirect if not logged in
if (!isLoggedIn()) {
    setFlashMessage('danger', 'Please log in to access this page');
    redirect(APP_URL . '/pages/login.php');
}

$pageTitle = 'Analytics';
$userId = $_SESSION['user_id'];

// Get user data
$user = getUserById($userId);

// Date range
$startDate = isset($_GET['start_date']) ? sanitize($_GET['start_date']) : date('Y-m-d', strtotime('-30 days'));
$endDate = isset($_GET['end_date']) ? sanitize($_GET['end_date']) : date('Y-m-d');

// Get user's books
$books = $db->select("SELECT * FROM books WHERE user_id = ? ORDER BY created_at DESC", [$userId]);
$bookIds = [];
foreach ($books as $book) {
    $bookIds[] = $book['id'];
}

// If user has no books, show message
if (empty($bookIds)) {
    $hasBooks = false;
} else {
    $hasBooks = true;
    $bookIdsStr = implode(',', $bookIds);
    
    // Get page views for this user's books
    $pageViewsQuery = "
        SELECT DATE(visit_date) as date, COUNT(*) as count 
        FROM page_views 
        WHERE page_type = 'book' AND page_id IN ($bookIdsStr)
        AND visit_date BETWEEN ? AND ? 
        GROUP BY DATE(visit_date) 
        ORDER BY date
    ";
    $pageViews = $db->select($pageViewsQuery, [$startDate, $endDate . ' 23:59:59']);
    
    // Get total page views
    $totalViewsQuery = "
        SELECT COUNT(*) as count 
        FROM page_views 
        WHERE page_type = 'book' AND page_id IN ($bookIdsStr)
    ";
    $totalViews = $db->selectOne($totalViewsQuery)['count'];
    
    // Get views by book
    $viewsByBookQuery = "
        SELECT b.id, b.title, COUNT(pv.id) as views
        FROM books b
        LEFT JOIN page_views pv ON pv.page_id = b.id AND pv.page_type = 'book'
        WHERE b.user_id = ?
        GROUP BY b.id
        ORDER BY views DESC
    ";
    $viewsByBook = $db->select($viewsByBookQuery, [$userId]);
    
    // Get link clicks
    $linkClicksQuery = "
        SELECT DATE(lc.visit_date) as date, COUNT(*) as count 
        FROM link_clicks lc
        JOIN buttons bt ON lc.button_id = bt.id
        JOIN books b ON bt.book_id = b.id
        WHERE b.user_id = ?
        AND lc.visit_date BETWEEN ? AND ? 
        GROUP BY DATE(lc.visit_date) 
        ORDER BY date
    ";
    $linkClicks = $db->select($linkClicksQuery, [$userId, $startDate, $endDate . ' 23:59:59']);
    
    // Get total link clicks
    $totalClicksQuery = "
        SELECT COUNT(*) as count 
        FROM link_clicks lc
        JOIN buttons bt ON lc.button_id = bt.id
        JOIN books b ON bt.book_id = b.id
        WHERE b.user_id = ?
    ";
    $totalClicks = $db->selectOne($totalClicksQuery, [$userId])['count'];
    
    // Get most clicked links
    $topLinksQuery = "
        SELECT bt.id, bt.label, bt.url, b.title, COUNT(lc.id) as clicks
        FROM buttons bt
        JOIN books b ON bt.book_id = b.id
        LEFT JOIN link_clicks lc ON lc.button_id = bt.id
        WHERE b.user_id = ?
        GROUP BY bt.id
        ORDER BY clicks DESC
        LIMIT 10
    ";
    $topLinks = $db->select($topLinksQuery, [$userId]);
    
    // Get referrers (where visitors came from)
    $referrersQuery = "
        SELECT referrer, COUNT(*) as count
        FROM page_views
        WHERE page_type = 'book' AND page_id IN ($bookIdsStr)
        AND visit_date BETWEEN ? AND ?
        AND referrer IS NOT NULL AND referrer != ''
        GROUP BY referrer
        ORDER BY count DESC
        LIMIT 5
    ";
    $referrers = $db->select($referrersQuery, [$startDate, $endDate . ' 23:59:59']);
}

include '../templates/header.php';
?>

<div class="row mb-4">
    <div class="col">
        <h1>Analytics</h1>
        <p>View statistics for your book pages and links</p>
    </div>
    <?php if ($hasBooks): ?>
    <div class="col-auto">
        <form class="d-flex" action="">
            <div class="input-group me-2">
                <span class="input-group-text">From</span>
                <input type="date" name="start_date" class="form-control" value="<?php echo $startDate; ?>">
            </div>
            <div class="input-group me-2">
                <span class="input-group-text">To</span>
                <input type="date" name="end_date" class="form-control" value="<?php echo $endDate; ?>">
            </div>
            <button type="submit" class="btn btn-primary">Apply</button>
        </form>
    </div>
    <?php endif; ?>
</div>

<?php if (!$hasBooks): ?>
    <div class="alert alert-info">
        <p>You haven't added any books yet. <a href="<?php echo APP_URL; ?>/pages/edit-book.php">Add your first book</a> to start tracking analytics.</p>
    </div>
<?php else: ?>

<!-- Statistics Cards -->
<div class="row">
    <div class="col-md-6 mb-4">
        <div class="card h-100 border-left-primary">
            <div class="card-body">
                <div class="row no-gutters align-items-center">
                    <div class="col mr-2">
                        <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Total Page Views</div>
                        <div class="h5 mb-0 font-weight-bold"><?php echo number_format($totalViews); ?></div>
                    </div>
                    <div class="col-auto">
                        <i class="fas fa-eye fa-2x text-gray-300"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-md-6 mb-4">
        <div class="card h-100 border-left-success">
            <div class="card-body">
                <div class="row no-gutters align-items-center">
                    <div class="col mr-2">
                        <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Total Link Clicks</div>
                        <div class="h5 mb-0 font-weight-bold"><?php echo number_format($totalClicks); ?></div>
                    </div>
                    <div class="col-auto">
                        <i class="fas fa-link fa-2x text-gray-300"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Charts -->
<div class="row">
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="m-0 font-weight-bold">Page Views</h5>
            </div>
            <div class="card-body">
                <canvas id="pageViewsChart" width="100%" height="300"></canvas>
            </div>
        </div>
    </div>
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="m-0 font-weight-bold">Link Clicks</h5>
            </div>
            <div class="card-body">
                <canvas id="linkClicksChart" width="100%" height="300"></canvas>
            </div>
        </div>
    </div>
</div>

<!-- Content Row -->
<div class="row">
    <!-- Books by Views -->
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="m-0 font-weight-bold">Books by Views</h5>
            </div>
            <div class="card-body">
                <?php if (count($viewsByBook) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                            <thead>
                                <tr>
                                    <th>Title</th>
                                    <th>Views</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($viewsByBook as $book): ?>
                                    <tr>
                                        <td><?php echo sanitize($book['title']); ?></td>
                                        <td><?php echo number_format($book['views']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p class="text-center text-muted">No data available yet</p>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Top Clicked Links -->
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="m-0 font-weight-bold">Top Clicked Links</h5>
            </div>
            <div class="card-body">
                <?php if (count($topLinks) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                            <thead>
                                <tr>
                                    <th>Link</th>
                                    <th>Book</th>
                                    <th>Clicks</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($topLinks as $link): ?>
                                    <tr>
                                        <td>
                                            <a href="<?php echo sanitize($link['url']); ?>" target="_blank">
                                                <?php echo sanitize($link['label']); ?>
                                            </a>
                                        </td>
                                        <td><?php echo sanitize($link['title']); ?></td>
                                        <td><?php echo number_format($link['clicks']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p class="text-center text-muted">No data available yet</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Referrers -->
<div class="row">
    <div class="col-lg-12">
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="m-0 font-weight-bold">Top Referrers</h5>
            </div>
            <div class="card-body">
                <?php if (count($referrers) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                            <thead>
                                <tr>
                                    <th>Referrer</th>
                                    <th>Count</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($referrers as $referrer): ?>
                                    <tr>
                                        <td><?php echo empty($referrer['referrer']) ? 'Direct / None' : sanitize($referrer['referrer']); ?></td>
                                        <td><?php echo number_format($referrer['count']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p class="text-center text-muted">No referrer data available yet</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js@3.7.1/dist/chart.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Prepare data for the page views chart
    const pageViewsData = {
        labels: [
            <?php
            $dates = [];
            $views = [];
            
            // Create date range array
            $current = strtotime($startDate);
            $end = strtotime($endDate);
            
            while ($current <= $end) {
                $currentDate = date('Y-m-d', $current);
                $dates[] = "'" . date('M j', $current) . "'";
                
                // Find views for this date
                $found = false;
                foreach ($pageViews as $view) {
                    if ($view['date'] == $currentDate) {
                        $views[] = $view['count'];
                        $found = true;
                        break;
                    }
                }
                
                if (!$found) {
                    $views[] = 0;
                }
                
                $current = strtotime('+1 day', $current);
            }
            
            echo implode(', ', $dates);
            ?>
        ],
        datasets: [{
            label: 'Page Views',
            data: [<?php echo implode(', ', $views); ?>],
            backgroundColor: 'rgba(78, 115, 223, 0.05)',
            borderColor: 'rgba(78, 115, 223, 1)',
            pointRadius: 3,
            pointBackgroundColor: 'rgba(78, 115, 223, 1)',
            pointBorderColor: 'rgba(78, 115, 223, 1)',
            pointHoverRadius: 3,
            pointHoverBackgroundColor: 'rgba(78, 115, 223, 1)',
            pointHoverBorderColor: 'rgba(78, 115, 223, 1)',
            pointHitRadius: 10,
            pointBorderWidth: 2,
            lineTension: 0.3
        }]
    };
    
    // Prepare data for the link clicks chart
    const linkClicksData = {
        labels: [
            <?php
            $clicks = [];
            
            // Reuse dates from page views
            // Create date range array
            $current = strtotime($startDate);
            $end = strtotime($endDate);
            
            while ($current <= $end) {
                $currentDate = date('Y-m-d', $current);
                
                // Find clicks for this date
                $found = false;
                foreach ($linkClicks as $click) {
                    if ($click['date'] == $currentDate) {
                        $clicks[] = $click['count'];
                        $found = true;
                        break;
                    }
                }
                
                if (!$found) {
                    $clicks[] = 0;
                }
                
                $current = strtotime('+1 day', $current);
            }
            
            // Dates are already echoed in page views chart
            echo implode(', ', $dates);
            ?>
        ],
        datasets: [{
            label: 'Link Clicks',
            data: [<?php echo implode(', ', $clicks); ?>],
            backgroundColor: 'rgba(28, 200, 138, 0.05)',
            borderColor: 'rgba(28, 200, 138, 1)',
            pointRadius: 3,
            pointBackgroundColor: 'rgba(28, 200, 138, 1)',
            pointBorderColor: 'rgba(28, 200, 138, 1)',
            pointHoverRadius: 3,
            pointHoverBackgroundColor: 'rgba(28, 200, 138, 1)',
            pointHoverBorderColor: 'rgba(28, 200, 138, 1)',
            pointHitRadius: 10,
            pointBorderWidth: 2,
            lineTension: 0.3
        }]
    };
    
    // Draw page views chart
    const pageViewsCtx = document.getElementById('pageViewsChart');
    if (pageViewsCtx) {
        const pageViewsChart = new Chart(pageViewsCtx, {
            type: 'line',
            data: pageViewsData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            precision: 0
                        }
                    }
                }
            }
        });
    }
    
    // Draw link clicks chart
    const linkClicksCtx = document.getElementById('linkClicksChart');
    if (linkClicksCtx) {
        const linkClicksChart = new Chart(linkClicksCtx, {
            type: 'line',
            data: linkClicksData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            precision: 0
                        }
                    }
                }
            }
        });
    }
});
</script>

<?php endif; ?>

<?php include '../templates/footer.php'; ?>