<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    redirect(APP_URL . '/admin/login.php');
}

$pageTitle = 'Manage Users';
$adminId = $_SESSION['admin_id'];
$success = '';
$error = '';

// Handle user actions (delete, suspend, etc.)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && isset($_POST['user_id'])) {
        $userId = (int)$_POST['user_id'];
        $action = $_POST['action'];
        
        // Make sure user exists
        $user = getUserById($userId);
        
        if ($user) {
            if ($action === 'delete') {
                // Delete user
                $db->delete('users', 'id = ?', [$userId]);
                $success = 'User deleted successfully';
            } elseif ($action === 'reset_password') {
                // Reset password to a random string
                $newPassword = generateRandomString(10);
                $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
                
                $db->update('users', ['password' => $hashedPassword], 'id = ?', [$userId]);
                $success = 'Password reset to: ' . $newPassword;
            }
        } else {
            $error = 'User not found';
        }
    }
}

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Get total users count
$totalUsers = $db->selectOne("SELECT COUNT(*) as count FROM users")['count'];
$totalPages = ceil($totalUsers / $perPage);

// Get users with pagination
$users = $db->select("
    SELECT u.*, 
           COUNT(DISTINCT b.id) as book_count,
           MAX(u.created_at) as created
    FROM users u
    LEFT JOIN books b ON u.id = b.user_id
    GROUP BY u.id
    ORDER BY u.id DESC
    LIMIT $perPage OFFSET $offset
");

// Include header
include 'templates/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h1 class="h3">Manage Users</h1>
    
    <div class="d-flex">
        <form class="d-flex me-2" action="">
            <input type="text" name="search" class="form-control me-2" placeholder="Search users...">
            <button type="submit" class="btn btn-primary">Search</button>
        </form>
    </div>
</div>

<?php if (!empty($success)): ?>
    <div class="alert alert-success">
        <?php echo $success; ?>
    </div>
<?php endif; ?>

<?php if (!empty($error)): ?>
    <div class="alert alert-danger">
        <?php echo $error; ?>
    </div>
<?php endif; ?>

<div class="card">
    <div class="card-header">
        <h6 class="m-0 font-weight-bold">Users</h6>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Username</th>
                        <th>Email</th>
                        <th>Permalink</th>
                        <th>Books</th>
                        <th>Created</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $user): ?>
                        <tr>
                            <td><?php echo $user['id']; ?></td>
                            <td><?php echo sanitize($user['username']); ?></td>
                            <td><?php echo sanitize($user['email']); ?></td>
                            <td>
                                <a href="<?php echo APP_URL . '/' . sanitize($user['permalink']); ?>" target="_blank">
                                    <?php echo sanitize($user['permalink']); ?>
                                </a>
                            </td>
                            <td><?php echo $user['book_count']; ?></td>
                            <td><?php echo date('M j, Y', strtotime($user['created'])); ?></td>
                            <td>
                                <div class="btn-group">
                                    <a href="user-details.php?id=<?php echo $user['id']; ?>" class="btn btn-sm btn-info">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <button type="button" class="btn btn-sm btn-warning" data-bs-toggle="modal" data-bs-target="#resetPasswordModal<?php echo $user['id']; ?>">
                                        <i class="fas fa-key"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-danger" data-bs-toggle="modal" data-bs-target="#deleteUserModal<?php echo $user['id']; ?>">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                                
                                <!-- Reset Password Modal -->
                                <div class="modal fade" id="resetPasswordModal<?php echo $user['id']; ?>" tabindex="-1" aria-hidden="true">
                                    <div class="modal-dialog">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <h5 class="modal-title">Reset Password</h5>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                            </div>
                                            <div class="modal-body">
                                                <p>Are you sure you want to reset the password for user <strong><?php echo sanitize($user['username']); ?></strong>?</p>
                                                <p>A new random password will be generated.</p>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                <form method="POST">
                                                    <input type="hidden" name="action" value="reset_password">
                                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                    <button type="submit" class="btn btn-warning">Reset Password</button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Delete User Modal -->
                                <div class="modal fade" id="deleteUserModal<?php echo $user['id']; ?>" tabindex="-1" aria-hidden="true">
                                    <div class="modal-dialog">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <h5 class="modal-title">Delete User</h5>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                            </div>
                                            <div class="modal-body">
                                                <p>Are you sure you want to delete user <strong><?php echo sanitize($user['username']); ?></strong>?</p>
                                                <p class="text-danger">This action cannot be undone and will delete all books and links associated with this user.</p>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                <form method="POST">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                    <button type="submit" class="btn btn-danger">Delete User</button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
            <nav aria-label="Page navigation">
                <ul class="pagination justify-content-center mt-4">
                    <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $page - 1; ?>" aria-label="Previous">
                            <span aria-hidden="true">&laquo;</span>
                        </a>
                    </li>
                    
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                        <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                        </li>
                    <?php endfor; ?>
                    
                    <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $page + 1; ?>" aria-label="Next">
                            <span aria-hidden="true">&raquo;</span>
                        </a>
                    </li>
                </ul>
            </nav>
        <?php endif; ?>
    </div>
</div>

<?php include 'templates/footer.php'; ?>