<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    redirect(APP_URL . '/admin/login.php');
}

// Check if super admin
if (!isset($_SESSION['is_super_admin']) || !$_SESSION['is_super_admin']) {
    setFlashMessage('danger', 'You do not have permission to access this page');
    redirect(APP_URL . '/admin/dashboard.php');
}

$pageTitle = 'Settings';
$adminId = $_SESSION['admin_id'];
$success = '';
$error = '';

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];
        
        if ($action === 'update_settings') {
            // Here you would update your site settings
            // This is a placeholder since we don't have a settings table yet
            
            $success = 'Settings updated successfully';
        } elseif ($action === 'add_admin') {
            $username = sanitize($_POST['username']);
            $email = sanitize($_POST['email']);
            $password = $_POST['password'];
            $confirmPassword = $_POST['confirm_password'];
            $isSuperAdmin = isset($_POST['is_super_admin']) ? 1 : 0;
            
            // Validate input
            if (empty($username)) {
                $error = 'Username is required';
            } elseif (empty($email)) {
                $error = 'Email is required';
            } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $error = 'Invalid email format';
            } elseif (empty($password)) {
                $error = 'Password is required';
            } elseif (strlen($password) < 8) {
                $error = 'Password must be at least 8 characters';
            } elseif ($password !== $confirmPassword) {
                $error = 'Passwords do not match';
            } else {
                // Check if username or email already exists
                $existingAdmin = $db->selectOne("SELECT * FROM admins WHERE username = ? OR email = ?", [$username, $email]);
                
                if ($existingAdmin) {
                    if ($existingAdmin['username'] === $username) {
                        $error = 'Username already exists';
                    } else {
                        $error = 'Email already exists';
                    }
                } else {
                    // Add admin
                    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                    
                    $adminData = [
                        'username' => $username,
                        'email' => $email,
                        'password' => $hashedPassword,
                        'is_super_admin' => $isSuperAdmin
                    ];
                    
                    $db->insert('admins', $adminData);
                    $success = 'Admin added successfully';
                }
            }
        } elseif ($action === 'delete_admin' && isset($_POST['admin_id'])) {
            $targetAdminId = (int)$_POST['admin_id'];
            
            // Make sure not deleting own account
            if ($targetAdminId === $adminId) {
                $error = 'You cannot delete your own account';
            } else {
                $db->delete('admins', 'id = ?', [$targetAdminId]);
                $success = 'Admin deleted successfully';
            }
        }
    }
}

// Get all admins
$admins = $db->select("SELECT * FROM admins ORDER BY id");

// Include header
include 'templates/header.php';
?>

<h1 class="h3 mb-4">Settings</h1>

<?php if (!empty($success)): ?>
    <div class="alert alert-success">
        <?php echo $success; ?>
    </div>
<?php endif; ?>

<?php if (!empty($error)): ?>
    <div class="alert alert-danger">
        <?php echo $error; ?>
    </div>
<?php endif; ?>

<!-- Admin Management -->
<div class="card mb-4">
    <div class="card-header">
        <h6 class="m-0 font-weight-bold">Admin Users</h6>
    </div>
    <div class="card-body">
        <div class="table-responsive mb-4">
            <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Username</th>
                        <th>Email</th>
                        <th>Role</th>
                        <th>Created</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($admins as $admin): ?>
                        <tr>
                            <td><?php echo $admin['id']; ?></td>
                            <td><?php echo sanitize($admin['username']); ?></td>
                            <td><?php echo sanitize($admin['email']); ?></td>
                            <td>
                                <?php if ($admin['is_super_admin']): ?>
                                    <span class="badge bg-danger">Super Admin</span>
                                <?php else: ?>
                                    <span class="badge bg-primary">Admin</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo date('M j, Y', strtotime($admin['created_at'])); ?></td>
                            <td>
                                <?php if ($admin['id'] !== $adminId): ?>
                                    <form method="POST" class="d-inline">
                                        <input type="hidden" name="action" value="delete_admin">
                                        <input type="hidden" name="admin_id" value="<?php echo $admin['id']; ?>">
                                        <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this admin?')">
                                            <i class="fas fa-trash"></i> Delete
                                        </button>
                                    </form>
                                <?php else: ?>
                                    <span class="text-muted">Current User</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <h5>Add New Admin</h5>
        <form method="POST" class="row g-3">
            <input type="hidden" name="action" value="add_admin">
            
            <div class="col-md-6">
                <label for="username" class="form-label">Username</label>
                <input type="text" id="username" name="username" class="form-control" required>
            </div>
            
            <div class="col-md-6">
                <label for="email" class="form-label">Email</label>
                <input type="email" id="email" name="email" class="form-control" required>
            </div>
            
            <div class="col-md-6">
                <label for="password" class="form-label">Password</label>
                <input type="password" id="password" name="password" class="form-control" required>
            </div>
            
            <div class="col-md-6">
                <label for="confirm_password" class="form-label">Confirm Password</label>
                <input type="password" id="confirm_password" name="confirm_password" class="form-control" required>
            </div>
            
            <div class="col-12">
                <div class="form-check">
                    <input type="checkbox" id="is_super_admin" name="is_super_admin" class="form-check-input">
                    <label for="is_super_admin" class="form-check-label">Super Admin</label>
                </div>
            </div>
            
            <div class="col-12">
                <button type="submit" class="btn btn-primary">Add Admin</button>
            </div>
        </form>
    </div>
</div>

<!-- Site Settings -->
<div class="card mb-4">
    <div class="card-header">
        <h6 class="m-0 font-weight-bold">Site Settings</h6>
    </div>
    <div class="card-body">
        <form method="POST">
            <input type="hidden" name="action" value="update_settings">
            
            <div class="mb-3">
                <label for="site_name" class="form-label">Site Name</label>
                <input type="text" id="site_name" name="site_name" class="form-control" value="<?php echo SITE_NAME; ?>">
            </div>
            
            <div class="mb-3">
                <label for="app_url" class="form-label">Site URL</label>
                <input type="text" id="app_url" name="app_url" class="form-control" value="<?php echo APP_URL; ?>">
            </div>
            
            <div class="mb-3">
                <label for="max_books_per_user" class="form-label">Max Books Per User</label>
                <input type="number" id="max_books_per_user" name="max_books_per_user" class="form-control" value="10">
            </div>
            
            <div class="mb-3">
                <label for="max_links_per_book" class="form-label">Max Links Per Book</label>
                <input type="number" id="max_links_per_book" name="max_links_per_book" class="form-control" value="20">
            </div>
            
            <button type="submit" class="btn btn-primary">Save Settings</button>
        </form>
    </div>
</div>

<?php include 'templates/footer.php'; ?>