<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    redirect(APP_URL . '/admin/login.php');
}

$pageTitle = 'Manage Books';
$adminId = $_SESSION['admin_id'];
$success = '';
$error = '';

// Handle book actions (delete, unpublish, etc.)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && isset($_POST['book_id'])) {
        $bookId = (int)$_POST['book_id'];
        $action = $_POST['action'];
        
        // Make sure book exists
        $book = getBookById($bookId);
        
        if ($book) {
            if ($action === 'delete') {
                // Delete book
                $db->delete('books', 'id = ?', [$bookId]);
                $success = 'Book deleted successfully';
            } elseif ($action === 'toggle_publish') {
                // Toggle publish status
                $newStatus = $book['is_published'] ? 0 : 1;
                $db->update('books', ['is_published' => $newStatus], 'id = ?', [$bookId]);
                $success = 'Book ' . ($newStatus ? 'published' : 'unpublished') . ' successfully';
            }
        } else {
            $error = 'Book not found';
        }
    }
}

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Filter
$filter = isset($_GET['filter']) ? $_GET['filter'] : '';
$filterSql = '';
$filterParams = [];

if ($filter === 'published') {
    $filterSql = 'AND b.is_published = 1';
} elseif ($filter === 'unpublished') {
    $filterSql = 'AND b.is_published = 0';
}

// Search
$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';
$searchSql = '';
$searchParams = [];

if (!empty($search)) {
    $searchSql = 'AND (b.title LIKE ? OR u.username LIKE ?)';
    $searchParams = ["%$search%", "%$search%"];
}

// Get total books count
$countQuery = "
    SELECT COUNT(*) as count 
    FROM books b
    JOIN users u ON b.user_id = u.id
    WHERE 1=1 $filterSql $searchSql
";
$countParams = array_merge($filterParams, $searchParams);
$totalBooks = $db->selectOne($countQuery, $countParams)['count'];
$totalPages = ceil($totalBooks / $perPage);

// Get books with pagination
$booksQuery = "
    SELECT b.*, u.username, u.permalink,
           (SELECT COUNT(*) FROM buttons WHERE book_id = b.id) as button_count
    FROM books b
    JOIN users u ON b.user_id = u.id
    WHERE 1=1 $filterSql $searchSql
    ORDER BY b.id DESC
    LIMIT $perPage OFFSET $offset
";
$booksParams = array_merge($filterParams, $searchParams);
$books = $db->select($booksQuery, $booksParams);

// Include header
include 'templates/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h1 class="h3">Manage Books</h1>
    
    <div class="d-flex">
        <form class="d-flex me-2" action="">
            <input type="text" name="search" class="form-control me-2" placeholder="Search books..." value="<?php echo sanitize($search); ?>">
            <button type="submit" class="btn btn-primary">Search</button>
        </form>
        
        <div class="dropdown ms-2">
            <button class="btn btn-secondary dropdown-toggle" type="button" id="filterDropdown" data-bs-toggle="dropdown" aria-expanded="false">
                Filter
            </button>
            <ul class="dropdown-menu" aria-labelledby="filterDropdown">
                <li><a class="dropdown-item <?php echo $filter === '' ? 'active' : ''; ?>" href="?">All Books</a></li>
                <li><a class="dropdown-item <?php echo $filter === 'published' ? 'active' : ''; ?>" href="?filter=published">Published</a></li>
                <li><a class="dropdown-item <?php echo $filter === 'unpublished' ? 'active' : ''; ?>" href="?filter=unpublished">Unpublished</a></li>
            </ul>
        </div>
    </div>
</div>

<?php if (!empty($success)): ?>
    <div class="alert alert-success">
        <?php echo $success; ?>
    </div>
<?php endif; ?>

<?php if (!empty($error)): ?>
    <div class="alert alert-danger">
        <?php echo $error; ?>
    </div>
<?php endif; ?>

<div class="card">
    <div class="card-header">
        <h6 class="m-0 font-weight-bold">Books</h6>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Title</th>
                        <th>Author</th>
                        <th>Links</th>
                        <th>Status</th>
                        <th>Created</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($books as $book): ?>
                        <tr>
                            <td><?php echo $book['id']; ?></td>
                            <td><?php echo sanitize($book['title']); ?></td>
                            <td>
                                <a href="user-details.php?id=<?php echo $book['user_id']; ?>">
                                    <?php echo sanitize($book['username']); ?>
                                </a>
                            </td>
                            <td><?php echo $book['button_count']; ?></td>
                            <td>
                                <?php if ($book['is_published']): ?>
                                    <span class="badge bg-success">Published</span>
                                <?php else: ?>
                                    <span class="badge bg-secondary">Unpublished</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo date('M j, Y', strtotime($book['created_at'])); ?></td>
                            <td>
                                <div class="btn-group">
                                    <a href="book-details.php?id=<?php echo $book['id']; ?>" class="btn btn-sm btn-info">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <a href="<?php echo APP_URL . '/' . sanitize($book['permalink']); ?>" target="_blank" class="btn btn-sm btn-primary">
                                        <i class="fas fa-external-link-alt"></i>
                                    </a>
                                    <button type="button" class="btn btn-sm btn-<?php echo $book['is_published'] ? 'warning' : 'success'; ?>" data-bs-toggle="modal" data-bs-target="#togglePublishModal<?php echo $book['id']; ?>">
                                        <i class="fas fa-<?php echo $book['is_published'] ? 'eye-slash' : 'eye'; ?>"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-danger" data-bs-toggle="modal" data-bs-target="#deleteBookModal<?php echo $book['id']; ?>">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                                
                                <!-- Toggle Publish Modal -->
                                <div class="modal fade" id="togglePublishModal<?php echo $book['id']; ?>" tabindex="-1" aria-hidden="true">
                                    <div class="modal-dialog">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <h5 class="modal-title"><?php echo $book['is_published'] ? 'Unpublish' : 'Publish'; ?> Book</h5>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                            </div>
                                            <div class="modal-body">
                                                <p>Are you sure you want to <?php echo $book['is_published'] ? 'unpublish' : 'publish'; ?> the book <strong><?php echo sanitize($book['title']); ?></strong>?</p>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                <form method="POST">
                                                    <input type="hidden" name="action" value="toggle_publish">
                                                    <input type="hidden" name="book_id" value="<?php echo $book['id']; ?>">
                                                    <button type="submit" class="btn btn-<?php echo $book['is_published'] ? 'warning' : 'success'; ?>">
                                                        <?php echo $book['is_published'] ? 'Unpublish' : 'Publish'; ?>
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Delete Book Modal -->
                                <div class="modal fade" id="deleteBookModal<?php echo $book['id']; ?>" tabindex="-1" aria-hidden="true">
                                    <div class="modal-dialog">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <h5 class="modal-title">Delete Book</h5>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                            </div>
                                            <div class="modal-body">
                                                <p>Are you sure you want to delete the book <strong><?php echo sanitize($book['title']); ?></strong>?</p>
                                                <p class="text-danger">This action cannot be undone and will delete all links associated with this book.</p>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                <form method="POST">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="book_id" value="<?php echo $book['id']; ?>">
                                                    <button type="submit" class="btn btn-danger">Delete Book</button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
            <nav aria-label="Page navigation">
                <ul class="pagination justify-content-center mt-4">
                    <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $page - 1; ?>&filter=<?php echo $filter; ?>&search=<?php echo urlencode($search); ?>" aria-label="Previous">
                            <span aria-hidden="true">&laquo;</span>
                        </a>
                    </li>
                    
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                        <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $i; ?>&filter=<?php echo $filter; ?>&search=<?php echo urlencode($search); ?>"><?php echo $i; ?></a>
                        </li>
                    <?php endfor; ?>
                    
                    <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $page + 1; ?>&filter=<?php echo $filter; ?>&search=<?php echo urlencode($search); ?>" aria-label="Next">
                            <span aria-hidden="true">&raquo;</span>
                        </a>
                    </li>
                </ul>
            </nav>
        <?php endif; ?>
    </div>
</div>

<?php include 'templates/footer.php'; ?>