<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    redirect(APP_URL . '/admin/login.php');
}

$pageTitle = 'Book Details';
$adminId = $_SESSION['admin_id'];
$error = '';

// Check if book ID is provided
if (!isset($_GET['id'])) {
    setFlashMessage('danger', 'Book ID is required');
    redirect(APP_URL . '/admin/books.php');
}

$bookId = (int)$_GET['id'];
$book = getBookById($bookId);

// Check if book exists
if (!$book) {
    setFlashMessage('danger', 'Book not found');
    redirect(APP_URL . '/admin/books.php');
}

// Get book author
$author = getUserById($book['user_id']);

// Get book buttons
$buttons = getButtonsByBookId($bookId);

// Get page views for this book
$pageViewsQuery = "
    SELECT DATE(visit_date) as date, COUNT(*) as count 
    FROM page_views 
    WHERE page_type = 'book' AND page_id = ?
    GROUP BY DATE(visit_date) 
    ORDER BY date DESC
    LIMIT 30
";
$pageViews = $db->select($pageViewsQuery, [$bookId]);

// Get link clicks for this book
$linkClicksQuery = "
    SELECT bt.id, bt.label, COUNT(lc.id) as clicks
    FROM buttons bt
    LEFT JOIN link_clicks lc ON lc.button_id = bt.id
    WHERE bt.book_id = ?
    GROUP BY bt.id
    ORDER BY clicks DESC
";
$linkClicks = $db->select($linkClicksQuery, [$bookId]);

// Include header
include 'templates/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h1 class="h3">Book Details</h1>
    
    <div>
        <a href="<?php echo APP_URL . '/' . sanitize($author['permalink']); ?>" target="_blank" class="btn btn-primary">
            <i class="fas fa-external-link-alt"></i> View Book Page
        </a>
        <a href="<?php echo APP_URL; ?>/admin/books.php" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> Back to Books
        </a>
    </div>
</div>

<div class="row">
    <div class="col-md-4">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Book Information</h6>
            </div>
            <div class="card-body">
                <div class="text-center mb-4">
                    <?php if (!empty($book['cover_image'])): ?>
                        <img src="<?php echo UPLOAD_URL . $book['cover_image']; ?>" alt="<?php echo sanitize($book['title']); ?>" class="img-fluid rounded" style="max-height: 250px;">
                    <?php else: ?>
                        <div class="bg-secondary p-5 text-white rounded">
                            <div>No Cover Image</div>
                        </div>
                    <?php endif; ?>
                </div>
                
                <h5><?php echo sanitize($book['title']); ?></h5>
                <p>
                    <strong>Author:</strong> 
                    <a href="user-details.php?id=<?php echo $book['user_id']; ?>">
                        <?php echo sanitize($author['username']); ?>
                    </a>
                </p>
                
                <p>
                    <strong>Status:</strong>
                    <?php if ($book['is_published']): ?>
                        <span class="badge bg-success">Published</span>
                    <?php else: ?>
                        <span class="badge bg-secondary">Unpublished</span>
                    <?php endif; ?>
                </p>
                
                <p><strong>Created:</strong> <?php echo date('M j, Y', strtotime($book['created_at'])); ?></p>
                <p><strong>Updated:</strong> <?php echo date('M j, Y', strtotime($book['updated_at'])); ?></p>
                
                <hr>
                
                <h6>Appearance Settings</h6>
                <div class="d-flex mb-2">
                    <div class="me-2">
                        <label>Background:</label>
                        <div style="width: 30px; height: 30px; background-color: <?php echo $book['background_color']; ?>; border: 1px solid #ddd;"></div>
                    </div>
                    <div class="me-2">
                        <label>Text:</label>
                        <div style="width: 30px; height: 30px; background-color: <?php echo $book['text_color']; ?>; border: 1px solid #ddd;"></div>
                    </div>
                    <div class="me-2">
                        <label>Buttons:</label>
                        <div style="width: 30px; height: 30px; background-color: <?php echo $book['button_color']; ?>; border: 1px solid #ddd;"></div>
                    </div>
                    <div>
                        <label>Button Text:</label>
                        <div style="width: 30px; height: 30px; background-color: <?php echo $book['button_text_color']; ?>; border: 1px solid #ddd;"></div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-8">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Book Description</h6>
            </div>
            <div class="card-body">
                <?php if (!empty($book['description'])): ?>
                    <?php echo nl2br(sanitize($book['description'])); ?>
                <?php else: ?>
                    <p class="text-muted">No description provided</p>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Book Links</h6>
            </div>
            <div class="card-body">
                <?php if (count($buttons) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                            <thead>
                                <tr>
                                    <th>Label</th>
                                    <th>URL</th>
                                    <th>Type</th>
                                    <th>Clicks</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($buttons as $button): 
                                    // Find clicks for this button
                                    $clicks = 0;
                                    foreach ($linkClicks as $click) {
                                        if ($click['id'] === $button['id']) {
                                            $clicks = $click['clicks'];
                                            break;
                                        }
                                    }
                                ?>
                                    <tr>
                                        <td><?php echo sanitize($button['label']); ?></td>
                                        <td>
                                            <a href="<?php echo sanitize($button['url']); ?>" target="_blank">
                                                <?php echo sanitize($button['url']); ?>
                                            </a>
                                        </td>
                                        <td><?php echo ucfirst(sanitize($button['button_type'])); ?></td>
                                        <td><?php echo $clicks; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p class="text-muted">No links added to this book</p>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Page Views (Last 30 Days)</h6>
            </div>
            <div class="card-body">
                <canvas id="pageViewsChart" width="100%" height="300"></canvas>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Page Views Chart
    const pageViewsCtx = document.getElementById('pageViewsChart');
    if (pageViewsCtx) {
        const pageViewsData = {
            labels: [
                <?php
                $dates = [];
                $views = [];
                
                // Get last 30 days
                for ($i = 29; $i >= 0; $i--) {
                    $date = date('Y-m-d', strtotime("-$i days"));
                    $dates[] = "'" . date('M j', strtotime($date)) . "'";
                    
                    // Find views for this date
                    $found = false;
                    foreach ($pageViews as $view) {
                        if ($view['date'] == $date) {
                            $views[] = $view['count'];
                            $found = true;
                            break;
                        }
                    }
                    
                    if (!$found) {
                        $views[] = 0;
                    }
                }
                
                echo implode(', ', $dates);
                ?>
            ],
            datasets: [{
                label: 'Page Views',
                data: [<?php echo implode(', ', $views); ?>],
                backgroundColor: 'rgba(78, 115, 223, 0.05)',
                borderColor: 'rgba(78, 115, 223, 1)',
                pointRadius: 3,
                pointBackgroundColor: 'rgba(78, 115, 223, 1)',
                pointBorderColor: 'rgba(78, 115, 223, 1)',
                pointHoverRadius: 3,
                pointHoverBackgroundColor: 'rgba(78, 115, 223, 1)',
                pointHoverBorderColor: 'rgba(78, 115, 223, 1)',
                pointHitRadius: 10,
                pointBorderWidth: 2,
                lineTension: 0.3
            }]
        };
        
        const pageViewsChart = new Chart(pageViewsCtx, {
            type: 'line',
            data: pageViewsData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            precision: 0
                        }
                    }
                }
            }
        });
    }
});
</script>

<?php include 'templates/footer.php'; ?>